﻿namespace Microsoft.Samples.PlanMyNight.Web.Controllers.Tests
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition.Hosting;
    using System.Linq;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Web.Controllers;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;
    using MvcContrib.TestHelper;

    [TestClass]
    public class SearchControllerFixture
    {
        public static void PrepareControllerMock(Controller controller)
        {
            var controllerContext = new Mock<ControllerContext>();
            var requestContext = new Mock<System.Web.Routing.RequestContext>();
            var routes = new System.Web.Routing.RouteCollection();
            var response = new Mock<System.Web.HttpResponseBase>();
            var request = new Mock<System.Web.HttpRequestBase>();
            var httpContext = new Mock<System.Web.HttpContextBase>();

            controllerContext.CallBase = true;

            requestContext.Setup(r => r.RouteData).Returns(new System.Web.Routing.RouteData());
            requestContext.Setup(r => r.HttpContext).Returns(httpContext.Object);

            httpContext.Setup(h => h.Response).Returns(response.Object);
            httpContext.Setup(h => h.Request).Returns(request.Object);

            controllerContext.Object.RequestContext = requestContext.Object;

            RoutingManager.RegisterRoutes(routes);

            response.Setup(r => r.ApplyAppPathModifier(It.IsAny<string>())).Returns(string.Empty);

            controller.Url = new UrlHelper(requestContext.Object, routes);
            controller.ControllerContext = controllerContext.Object;
        }
        
        [TestInitialize]
        public void InitializeServiceLocator()
        {
            ServiceLocator.SetLocatorProvider(() => new CompositionContainer(new TypeCatalog(typeof(MockActionExport))));
        }

        [TestMethod]
        public void IndexAddsActivitySearchFields()
        {
            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.RetrieveActivityTypes()).Returns(new[] { new ActivityType { Id = 1, Name = "Bar" }, new ActivityType { Id = 2, Name = "Restaurant" } });

            var reference = new Mock<IReferenceRepository>();
            reference.Setup(o => o.RetrieveStates())
                      .Returns(new[]
                      {
                          new State { Name = "Ohio", Abbreviation = "OH" },
                          new State { Name = "New York", Abbreviation = "NY" },
                          new State { Name = "Washington", Abbreviation = "WA" }
                      });

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, reference.Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Index();
            var viewData = result.AssertViewRendered().ViewData;

            result.AssertViewRendered();
            Assert.IsNotNull(viewData);
            var model = viewData.Model as SearchViewModel;
            Assert.IsNotNull(model);
            Assert.IsNotNull(model.SearchFields);
            Assert.IsNotNull(model.SearchFields.ActivityTypes);
            Assert.AreEqual(3, model.SearchFields.ActivityTypes.Count());
            Assert.IsNotNull(model.SearchFields.States);
            Assert.AreEqual(4, model.SearchFields.States.Count());
        }

        [TestMethod]
        public void SearchNoQueryReturnsModelStateError()
        {
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            builder.InitializeController(controller);

            var result = controller.Search(new NaturalSearchQuery { Query = string.Empty });

            var viewResult = result.AssertViewRendered();
            Assert.AreEqual("Index", viewResult.ViewName);
            Assert.IsFalse(viewResult.ViewData.ModelState.IsValid);
            Assert.AreEqual(1, viewResult.ViewData.ModelState["Query"].Errors.Count);
        }

        [TestMethod]
        public void SearchIncompleteQueryReturnsModelStateError()
        {
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            builder.InitializeController(controller);

            var result = controller.Search(new NaturalSearchQuery { Query = string.Empty });

            var viewResult = result.AssertViewRendered();
            Assert.AreEqual("Index", viewResult.ViewName);
            Assert.IsFalse(viewResult.ViewData.ModelState.IsValid);
            Assert.AreEqual(1, viewResult.ViewData.ModelState["Query"].Errors.Count);
        }

        [TestMethod]
        public void SearchInvalidActivityReturnsEmpty()
        {
            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });

            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repo.Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            builder.InitializeController(controller);

            var viewData = controller.Search(new NaturalSearchQuery { Query = "hotels in manhattan, ny" }).AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(model);
            Assert.IsNull(model.Items);
        }

        [TestMethod]
        public void SearchCallsRepository()
        {
            var expectedResult = new PagingResult<Activity>(new[] { new Activity { Id = "xxx" } })
            {
                CurrentPage = 1,
                PageSize = SearchController.DefaultPageSize,
                TotalItems = 1
            };

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });

            repo.Setup(o => o.Search(It.Is<NaturalSearchQuery>(s =>
                                                                        s.Query == "bars in manhattan, ny"
                                                                        && s.SortBy == SortCriteria.Relevance
                                                                        && s.Page == 1
                                                                        && s.PageSize == SearchController.DefaultPageSize)))
                .Returns(expectedResult);

            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repo.Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var viewData = controller.Search(new NaturalSearchQuery { Query = "bars in manhattan, ny", Page = 1 }).AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(model);
            var results = model.Items as IEnumerable<ActivityViewModel>;
            Assert.IsNotNull(results);
            Assert.AreEqual(1, results.Count());
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Activity, model.Criteria.Type);
        }

        [TestMethod]
        public void SearchForSpecificPageCallsRepository()
        {
            var expectedResult = new PagingResult<Activity>(new[] { new Activity { Id = "xxx" } })
            {
                CurrentPage = 2,
                PageSize = SearchController.DefaultPageSize,
                TotalItems = 12
            };

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });

            repo.Setup(o => o.Search(It.Is<NaturalSearchQuery>(s =>
                                                                        s.Query == "bars in manhattan, ny"
                                                                        && s.SortBy == SortCriteria.Relevance
                                                                        && s.Page == 2
                                                                        && s.PageSize == SearchController.DefaultPageSize)))
                .Returns(expectedResult);

            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repo.Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var viewData = controller.Search(new NaturalSearchQuery { Query = "bars in manhattan, ny", Page = 2 }).AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;
            var results = model.Items as IEnumerable<ActivityViewModel>;

            Assert.IsNotNull(results);
            Assert.AreEqual(2, model.CurrentPage);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Activity, model.Criteria.Type);
        }

        [TestMethod]
        public void SearchSetsDefaultPageSizeAndNumber()
        {
            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });

            repo.Setup(o => o.Search(It.Is<NaturalSearchQuery>(s =>
                                                                        s.Query == "bars in manhattan, ny"
                                                                        && s.ActivityTypeId == 11199
                                                                        && s.SortBy == SortCriteria.Relevance
                                                                        && s.Page == 1
                                                                        && s.PageSize == SearchController.DefaultPageSize)))
                .Returns(
                    new PagingResult<Activity>(new[] { new Activity { Id = "xxx" } })
                    {
                        CurrentPage = 1,
                        PageSize = SearchController.DefaultPageSize,
                        TotalItems = 1
                    });

            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repo.Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var viewData = controller.Search(new NaturalSearchQuery { Query = "bars in manhattan, ny" }).AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;
            var results = model.Items as IEnumerable<ActivityViewModel>;
            Assert.IsNotNull(results);
            Assert.AreEqual(1, model.CurrentPage);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Activity, model.Criteria.Type);
            Assert.AreEqual(1, model.Criteria.Page);
            Assert.AreEqual(SearchController.DefaultPageSize, model.Criteria.PageSize);
        }

        [TestMethod]
        public void SearchItinerariesParsesQueryAndCallsRepo()
        {
            var searchCriteria = new NaturalSearchQuery() { Query = "Bars in Colville, WA", Type = SearchType.Itinerary };
            var expectedResult = new PagingResult<Itinerary>(new[] { new Itinerary() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1
            };

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });
            activitiesRepository.Setup(o => o.ParseQueryLocation("Bars in Colville, WA"))
                                .Returns(new ActivityAddress { State = "WA", City = "Colville" });

            var itinerariesRepository = new Mock<IItinerariesRepository>();
            itinerariesRepository.Setup(o => o.SearchByCity(11199, "WA", "Colville", SearchController.DefaultPageSize, 1))
                      .Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, activitiesRepository.Object, itinerariesRepository.Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Search(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;

            activitiesRepository.VerifyAll();
            itinerariesRepository.VerifyAll();

            Assert.IsNotNull(viewData);
            var model = viewData.Model as SearchViewModel;
            var results = model.Items as IEnumerable<ItineraryViewModel>;
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Itinerary, model.Criteria.Type);
            Assert.IsNotNull(results);
        }

        [TestMethod]
        public void SearchItinerariesParsesNonValidQueryCallsRepoAndReturnsEmpty()
        {
            var searchCriteria = new NaturalSearchQuery() { Query = "bars in somewhere", Type = SearchType.Itinerary };

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.RetrieveActivityTypes())
                .Returns(new[]
                {
                    new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                    new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                    new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
                });
            activitiesRepository.Setup(o => o.ParseQueryLocation("bars in somewhere"))
                                .Returns(default(ActivityAddress));

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, activitiesRepository.Object, new Mock<IItinerariesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Search(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;

            Assert.IsNotNull(viewData);
            var model = viewData.Model as SearchViewModel;
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Itinerary, model.Criteria.Type);
            Assert.IsNull(model.Items);
            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void AdvancedSearchAddsActivitiesResults()
        {
            var searchCriteria = new AdvancedSearchQuery() { City = "Columbus", ActivityTypeId = 1, Type = SearchType.Activity };
            var expectedResult = new PagingResult<Activity>(new[] { new Activity() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1,
            };

            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.Search(It.Is<AdvancedSearchQuery>(s => s == searchCriteria))).Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model.Criteria);
            var results = model.Items as IEnumerable<ActivityViewModel>;
            Assert.AreEqual(SearchType.Activity, model.Criteria.Type);
            Assert.IsNotNull(results);
        }

        [TestMethod]
        public void AdvancedSearchRendersIndex()
        {
            var repository = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Advanced(new AdvancedSearchQuery { City = "Columbus", Type = SearchType.Activity });
            var viewResult = result.AssertViewRendered();
            Assert.AreEqual("Index", viewResult.ViewName);
        }

        [TestMethod]
        public void AdvancedSearchAddsLookupFieldsForSearch()
        {
            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.RetrieveActivityTypes()).Returns(new[] { new ActivityType { Id = 1, Name = "Bar" }, new ActivityType { Id = 2, Name = "Restaurant" } });

            var reference = new Mock<IReferenceRepository>();
            reference.Setup(o => o.RetrieveStates())
                      .Returns(new[]
                      {
                          new State { Name = "Ohio", Abbreviation = "OH" },
                          new State { Name = "New York", Abbreviation = "NY" },
                          new State { Name = "Washington", Abbreviation = "WA" }
                      });

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, reference.Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Advanced(new AdvancedSearchQuery());
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model);
            Assert.IsNotNull(model.SearchFields);
            Assert.IsNotNull(model.SearchFields.ActivityTypes);
            Assert.AreEqual(3, model.SearchFields.ActivityTypes.Count());
            Assert.IsNotNull(model.SearchFields.States);
            Assert.AreEqual(4, model.SearchFields.States.Count());
        }

        [TestMethod]
        public void AdvancedSearchSetsPageSizeAndCurrentPage()
        {
            var searchCriteria = new AdvancedSearchQuery()
                {
                    City = "Columbus",
                    ActivityTypeId = 1,
                    PageSize = 0,
                    Page = 0,
                    Type = SearchType.Activity
                };

            var expectedResult = new PagingResult<Activity>(new[] { new Activity() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1
            };

            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.Search(It.IsAny<AdvancedSearchQuery>())).Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            result.AssertViewRendered();

            repository.Verify(o => o.Search(It.Is<AdvancedSearchQuery>(s => s.PageSize != 0)));
            repository.Verify(o => o.Search(It.Is<AdvancedSearchQuery>(s => s.Page == 1)));
        }

        [TestMethod]
        public void AdvancedSearchSetsGetsFirstPageIfLessThan1IsSpecified()
        {
            var searchCriteria = new AdvancedSearchQuery()
            {
                City = "Columbus",
                ActivityTypeId = 1,
                Page = -1,
                Type = SearchType.Activity
            };

            var expectedResult = new PagingResult<Activity>(new[] { new Activity() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1,
            };
            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.Search(It.IsAny<AdvancedSearchQuery>())).Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            result.AssertViewRendered();

            repository.Verify(o => o.Search(It.Is<AdvancedSearchQuery>(s => s.Page == 1)));
        }

        [TestMethod]
        public void AdvancedSearchSetsTitleForSearchCriteria()
        {
            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.RetrieveActivityTypes()).Returns(new[] { new ActivityType { Id = 1, Name = "Bar", PluralizedName = "Bars" }, new ActivityType { Id = 2, Name = "Restaurant", PluralizedName = "Restaurants" } });

            var reference = new Mock<IReferenceRepository>();
            reference.Setup(o => o.RetrieveStates())
                      .Returns(new[]
                      {
                          new State { Name = "Ohio", Abbreviation = "OH" },
                          new State { Name = "New York", Abbreviation = "NY" },
                          new State { Name = "Washington", Abbreviation = "WA" }
                      });

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, reference.Object, repository.Object, new Mock<IItinerariesRepository>().Object);
            builder.InitializeController(controller);

            var searchCriteria = new AdvancedSearchQuery()
            {
                ActivityTypeId = 1,
                City = "Westerville",
                State = "OH",
                Zip = "43082"
            };

            var result = controller.Advanced(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(model);
            Assert.IsFalse(String.IsNullOrEmpty(model.CriteriaDescription));
            string description = model.CriteriaDescription;
            StringAssert.Contains(description, "Bars");
            StringAssert.Contains(description, "Westerville");
            StringAssert.Contains(description, "OH");
            StringAssert.Contains(description, "43082");
        }

        [TestMethod]
        public void AdvancedSearchItinerariesByStateAndCity()
        {
            var searchCriteria = new AdvancedSearchQuery() { ActivityTypeId = 1, State = "WA", City = "Colville", Type = SearchType.Itinerary };
            var expectedResult = new PagingResult<Itinerary>(new[] { new Itinerary() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1
            };

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.SearchByCity(1, "WA", "Colville", SearchController.DefaultPageSize, 1))
                      .Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object, repository.Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Itinerary, model.Criteria.Type);
            var results = model.Items as IEnumerable<ItineraryViewModel>;
            Assert.IsNotNull(results);
        }

        [TestMethod]
        public void AdvancedSearchItinerariesByStateAndCityAndStreetAddressAndRadiusAndZip()
        {
            var searchCriteria = new AdvancedSearchQuery() { ActivityTypeId = 1, State = "WA", City = "Colville", Type = SearchType.Itinerary, Radius = 1, StreetAddress = "350 Evergreen St.", Zip = "1000" };
            var expectedResult = new PagingResult<Itinerary>(new[] { new Itinerary() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1
            };

            var activities = new Mock<IActivitiesRepository>();
            var locations = new Tuple<double, double>(1, 2);

            activities.Setup(r => r.GeocodeAddress(It.Is<ActivityAddress>(aa => aa.City == searchCriteria.City && aa.State == searchCriteria.State && aa.StreetAddress == searchCriteria.StreetAddress && aa.Zip == searchCriteria.Zip))).Returns(locations);

            var repository = new Mock<IItinerariesRepository>();

            repository.Setup(o => o.SearchByRadius(1, 1, 2, It.IsAny<double>(), SearchController.DefaultPageSize, 1)).Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, activities.Object, repository.Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Itinerary, model.Criteria.Type);
            var results = model.Items as IEnumerable<ItineraryViewModel>;
            Assert.IsNotNull(results);
        }

        [TestMethod]
        public void AdvancedSearchItinerariesByZip()
        {
            var searchCriteria = new AdvancedSearchQuery() { ActivityTypeId = 1, Zip = "99999", Type = SearchType.Itinerary };
            var expectedResult = new PagingResult<Itinerary>(new[] { new Itinerary() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1,
            };

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.SearchByZipCode(1, "99999", SearchController.DefaultPageSize, 1))
                      .Returns(expectedResult);

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object, repository.Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.Advanced(searchCriteria);
            var viewData = result.AssertViewRendered().ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.Itinerary, model.Criteria.Type);
            var results = model.Items as IEnumerable<ItineraryViewModel>;
            Assert.IsNotNull(results);
        }

        [TestMethod]
        public void AdvancedSearchItinerariesByActivityCallsRepository()
        {
            var expectedResult = new PagingResult<Itinerary>(new[] { new Itinerary() })
            {
                TotalItems = 1,
                PageSize = SearchController.DefaultPageSize,
                CurrentPage = 1,
            };
            var repoItineraries = new Mock<IItinerariesRepository>();
            repoItineraries.Setup(o => o.SearchByActivity("1", SearchController.DefaultPageSize, 1))
                      .Returns(expectedResult);
            var repoActivities = new Mock<IActivitiesRepository>();
            repoActivities.Setup(o => o.RetrieveActivity("1"))
                          .Returns(new Activity());

            var builder = new TestControllerBuilder();
            var controller = new SearchController(new Mock<IMembershipService>().Object, new Mock<IReferenceRepository>().Object, repoActivities.Object, repoItineraries.Object);
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.ByActivity("1", null);
            var viewData = result.AssertViewRendered().ForView("Index").ViewData;
            var model = viewData.Model as SearchViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model.Criteria);
            Assert.AreEqual(SearchType.ActivityItinerary, model.Criteria.Type);
            var results = model.Items as IEnumerable<ItineraryViewModel>;
            Assert.IsNotNull(results);
            Assert.IsNotNull(model.SearchActivity);
        }

        [TestMethod]
        public void ZipCodesFiltersStartsWith()
        {
            var repo = new Mock<IReferenceRepository>();
            repo.Setup(o => o.RetrieveZipCodes("XX"))
                .Returns(new[] { "00001", "00002", "00003", "01000", "10000" });

            var controller = new SearchController(new Mock<IMembershipService>().Object, repo.Object, new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            PrepareControllerMock(controller);
            
            builder.InitializeController(controller);

            var result = controller.ZipCodes("XX", "000").AssertResultIs<JsonResult>();
            var zipCodes = result.Data as IEnumerable<string>;

            Assert.IsNotNull(zipCodes);
            Assert.AreEqual(3, zipCodes.Count());
            Assert.IsFalse(zipCodes.Any(s => s == "01000"));
            Assert.IsFalse(zipCodes.Any(s => s == "10000"));
        }

        [TestMethod]
        public void CitiesFiltersStartsWith()
        {
            var repo = new Mock<IReferenceRepository>();
            repo.Setup(o => o.RetrieveCities("XX"))
                .Returns(new[] { "City1", "City2", "Ciudad1", "stadt", "xxxx" });

            var controller = new SearchController(new Mock<IMembershipService>().Object, repo.Object, new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object);
            var builder = new TestControllerBuilder();
            PrepareControllerMock(controller);

            builder.InitializeController(controller);

            var result = controller.Cities("XX", "Ci").AssertResultIs<JsonResult>();
            var zipCodes = result.Data as IEnumerable<string>;

            Assert.IsNotNull(zipCodes);
            Assert.AreEqual(3, zipCodes.Count());
            Assert.IsFalse(zipCodes.Any(s => s == "stadt"));
            Assert.IsFalse(zipCodes.Any(s => s == "xxxx"));
        }
    }
}
